"""
Print registration renewal invoices for all vehicles.

Supports subclasses:
    Car
    Truck

Please note that this program uses a beginner approach that explicitly checks
for subtypes. If this were an intermediate Python course, we would
refactor this program and the classes in the hierarchy to eliminate explicit
subtype checking.
"""

from my_vehicles import Car, Truck


def main():
    vehicles = get_vehicles()
    vehicles.sort(key=by_first_name)
    vehicles.sort(key=by_last_name)
    vehicles.sort(key=by_city)
    print_invoices(vehicles)


def get_vehicles():
    infile_name = input('Please enter the input filename: ')
    with open(infile_name, 'r', encoding='utf-8') as infile:
        my_vehicles = []
        for line in infile:
            if line.startswith('Car'):
                vehicle = construct_car_instance(line)
            elif line.startswith('Truck'):
                vehicle = construct_truck_instance(line)
            else:
                raise ValueError(f'Car or Truck was expected. This line starts with: {line[0:10]}')
            my_vehicles.append(vehicle)
    return my_vehicles


def construct_car_instance(record):
    record = record.rstrip()
    data_fields = record.split(',')
    vehicle_type, first_name, last_name, street_address_1, street_address_2 = data_fields[0:5]
    city, state, zipcode, make, model, year, color, vehicle_id, fuel_type = data_fields[5:]
    vehicle = Car(first_name,
                  last_name,
                  street_address_1,
                  street_address_2,
                  city,
                  state,
                  zipcode,
                  make,
                  model,
                  int(year),
                  color,
                  vehicle_id,
                  fuel_type)
    return vehicle


def construct_truck_instance(record):
    record = record.rstrip()
    data_fields = record.split(',')
    vehicle_type, first_name, last_name, street_address_1, street_address_2 = data_fields[0:5]
    city, state, zipcode, make, model, year, color, vehicle_id, gross_weight = data_fields[5:]
    vehicle = Truck(first_name,
                    last_name,
                    street_address_1,
                    street_address_2,
                    city,
                    state,
                    zipcode,
                    make,
                    model,
                    int(year),
                    color,
                    vehicle_id,
                    int(gross_weight))
    return vehicle


def by_first_name(vehicle_instance):
    return vehicle_instance.first_name


def by_last_name(vehicle_instance):
    return vehicle_instance.last_name


def by_city(vehicle_instance):
    return vehicle_instance.city


def print_invoices(these_vehicles):
    invoices_printed = 0
    for vehicle in these_vehicles:
        print_invoice_for_this_vehicle(vehicle)
        invoices_printed += 1

    print_separator_line()
    print(f'\n\n{invoices_printed} invoices have been printed.')


def print_invoice_for_this_vehicle(vehicle):
    print_separator_line()
    print_invoice_title(vehicle)
    print_name_and_address_block(vehicle)
    print_common_data_fields(vehicle)
    print_subtype_specific_data_fields(vehicle)
    print_annual_fee_amount(vehicle)


def print_separator_line():
    print(f'\n\n{"-" * 45}')


def print_invoice_title(vehicle):
    if isinstance(vehicle, Car):
        subtype = 'Car'
    elif isinstance(vehicle, Truck):
        subtype = 'Truck'
    else:
        raise RuntimeError(f'Unexpected vehicle subtype found.')
    title = f'{subtype} Registration Renewal Invoice'
    print(f'\n{title.upper()}')
    print()


def print_name_and_address_block(vehicle):
    print(f'{vehicle.first_name} {vehicle.last_name}')
    print(vehicle.street_address_1)
    if vehicle.street_address_2:
        print(vehicle.street_address_2)
    print(f'{vehicle.city}, {vehicle.state} {vehicle.zipcode}')
    print()


def print_common_data_fields(vehicle):
    print(f'{"Make:":<20}    {vehicle.make}'),
    print(f'{"Model:":<20}    {vehicle.model}'),
    print(f'{"Year:":<20}    {vehicle.year}'),
    print(f'{"Color:":<20}    {vehicle.color}'),
    print(f'{"Vehicle ID:":<20}    {vehicle.vehicle_id}')


def print_subtype_specific_data_fields(vehicle):
    if isinstance(vehicle, Car):
        print_car_specific_data_fields(vehicle)
    elif isinstance(vehicle, Truck):
        print_truck_specific_data_fields(vehicle)
    else:
        raise RuntimeError(f'Unexpected vehicle subtype found.')


def print_car_specific_data_fields(vehicle):
    print(f'{"Fuel Type:":<20}    {vehicle.fuel_type}')


def print_truck_specific_data_fields(vehicle):
    print(f'{"Gross Weight:":<20}    {vehicle.gross_weight:,}')


def print_annual_fee_amount(vehicle):
    print()
    print(f'AMOUNT DUE: $ {vehicle.determine_annual_registration_fee():0.2f}')


main()
